/* OpenCP Module Player
 * copyright (c) 2020-'26 Stian Skjelstad <stian.skjelstad@gmail.com>
 *
 * Basic glue for the different console implementations for unix
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "config.h"
#include <errno.h>
#include <iconv.h>
#include <stdio.h>
#include <string.h>
#include "types.h"
#include "cp437.h"
#include "utf-8.h"

static iconv_t to_cp437_from_utf8 = (iconv_t)(-1);
static iconv_t from_cp437_to_utf8 = (iconv_t)(-1);

/* table to convert codepage 437 into Unicode (UTF-8) */
const uint32_t ocp_cp437_to_unicode[256] =
{
/*
	     0       1       2       3       4       5       6       7       8       9       a       b       c       d       e       f
*/
	0x000020, 0x00263a, 0x00263b, 0x002665, 0x200666, 0x002663, 0x002660, 0x002022, 0x0025d8, 0x25cb, 0x25d9, 0x2642, 0x2640, 0x00266a, 0x00266b, 0x00263c, // 0x
	0x0025ba, 0x0025c4, 0x002195, 0x00203c, 0x0000b6, 0x0000a7, 0x0025ac, 0x0021a8, 0x002191, 0x2193, 0x2192, 0x2190, 0x221f, 0x002194, 0x0025b2, 0x0025bc, // 1x
	0x000020, 0x000021, 0x000022, 0x000023, 0x000024, 0x000025, 0x000026, 0x000027, 0x000028, 0x0029, 0x002a, 0x002b, 0x002c, 0x00002d, 0x00002e, 0x00002f, // 2x
	0x000030, 0x000031, 0x000032, 0x000033, 0x000034, 0x000035, 0x000036, 0x000037, 0x000038, 0x0039, 0x003a, 0x003b, 0x003c, 0x00003d, 0x00003e, 0x00003f, // 3x
	0x000040, 0x000041, 0x000042, 0x000043, 0x000044, 0x000045, 0x000046, 0x000047, 0x000048, 0x0049, 0x004a, 0x004b, 0x004c, 0x00004d, 0x00004e, 0x00004f, // 4x
	0x000050, 0x000051, 0x000052, 0x000053, 0x000054, 0x000055, 0x000056, 0x000057, 0x000058, 0x0059, 0x005a, 0x005b, 0x005c, 0x00005d, 0x00005e, 0x00005f, // 5x
	0x000060, 0x000061, 0x000062, 0x000063, 0x000064, 0x000065, 0x000066, 0x000067, 0x000068, 0x0069, 0x006a, 0x006b, 0x006c, 0x00006d, 0x00006e, 0x00006f, // 6x
	0x000070, 0x000071, 0x000072, 0x000073, 0x000074, 0x000075, 0x000076, 0x000077, 0x000078, 0x0079, 0x007a, 0x007b, 0x00a6, 0x00007d, 0x00007e, 0x002302, // 7x
	0x0000c7, 0x0000fc, 0x0000e9, 0x0000e2, 0x0000e4, 0x0000e0, 0x0000e5, 0x0000e7, 0x0000ea, 0x00eb, 0x00e8, 0x00ef, 0x00ee, 0x0000ec, 0x0000c4, 0x0000c5, // 8x
	0x0000c9, 0x0000e6, 0x0000c6, 0x0000f4, 0x0000f6, 0x0000f2, 0x0000fb, 0x0000f9, 0x0000ff, 0x00d6, 0x00dc, 0x00a2, 0x00a3, 0x0000a5, 0x0020a7, 0x000192, // 9x
	0x0000e1, 0x0000ed, 0x0000f3, 0x0000fa, 0x0000f1, 0x0000d1, 0x0000aa, 0x0000ba, 0x0000bf, 0x2310, 0x00ac, 0x00bd, 0x00bc, 0x0000a1, 0x0000ab, 0x0000bb, // ax
	0x002591, 0x002592, 0x002593, 0x002502, 0x002524, 0x100000, 0x100001, 0x100002, 0x100003, 0x2563, 0x2551, 0x2557, 0x255d, 0x100004, 0x100005, 0x002510, // bx
	0x002514, 0x002534, 0x00252c, 0x00251c, 0x002500, 0x00253c, 0x100006, 0x100007, 0x00255a, 0x2554, 0x2569, 0x2566, 0x2560, 0x002550, 0x00256c, 0x100008, // cx
	0x100009, 0x10000a, 0x10000b, 0x10000c, 0x10000d, 0x10000e, 0x10000f, 0x100010, 0x00256a, 0x2518, 0x250c, 0x2588, 0x2584, 0x00258c, 0x002590, 0x002580, // dx
	0x0003b1, 0x0000df, 0x000393, 0x0003c0, 0x0003a3, 0x0003c3, 0x0000b5, 0x0003c4, 0x0003a6, 0x0398, 0x03a9, 0x03b4, 0x221e, 0x0003c6, 0x0003b5, 0x002229, // ex
	0x002261, 0x0000b1, 0x002265, 0x002264, 0x002320, 0x002321, 0x0000f7, 0x002248, 0x0000b0, 0x2219, 0x00b7, 0x221a, 0x207f, 0x0000b2, 0x0025a0, 0x0000a0  // fx
};

const uint16_t cp437_to_unicode[256] =
{
/*
	     0       1       2       3       4       5       6       7       8       9       a       b       c       d       e       f
*/
	0x0020, 0x263a, 0x263b, 0x2665, 0x2666, 0x2663, 0x2660, 0x2022, 0x25d8, 0x25cb, 0x25d9, 0x2642, 0x2640, 0x266a, 0x266b, 0x263c, // 0x
	0x25ba, 0x25c4, 0x2195, 0x203c, 0x00b6, 0x00a7, 0x25ac, 0x21a8, 0x2191, 0x2193, 0x2192, 0x2190, 0x221f, 0x2194, 0x25b2, 0x25bc, // 1x
	0x0020, 0x0021, 0x0022, 0x0023, 0x0024, 0x0025, 0x0026, 0x0027, 0x0028, 0x0029, 0x002a, 0x002b, 0x002c, 0x002d, 0x002e, 0x002f, // 2x
	0x0030, 0x0031, 0x0032, 0x0033, 0x0034, 0x0035, 0x0036, 0x0037, 0x0038, 0x0039, 0x003a, 0x003b, 0x003c, 0x003d, 0x003e, 0x003f, // 3x
	0x0040, 0x0041, 0x0042, 0x0043, 0x0044, 0x0045, 0x0046, 0x0047, 0x0048, 0x0049, 0x004a, 0x004b, 0x004c, 0x004d, 0x004e, 0x004f, // 4x
	0x0050, 0x0051, 0x0052, 0x0053, 0x0054, 0x0055, 0x0056, 0x0057, 0x0058, 0x0059, 0x005a, 0x005b, 0x005c, 0x005d, 0x005e, 0x005f, // 5x
	0x0060, 0x0061, 0x0062, 0x0063, 0x0064, 0x0065, 0x0066, 0x0067, 0x0068, 0x0069, 0x006a, 0x006b, 0x006c, 0x006d, 0x006e, 0x006f, // 6x
	0x0070, 0x0071, 0x0072, 0x0073, 0x0074, 0x0075, 0x0076, 0x0077, 0x0078, 0x0079, 0x007a, 0x007b, 0x00a6, 0x007d, 0x007e, 0x2302, // 7x
	0x00c7, 0x00fc, 0x00e9, 0x00e2, 0x00e4, 0x00e0, 0x00e5, 0x00e7, 0x00ea, 0x00eb, 0x00e8, 0x00ef, 0x00ee, 0x00ec, 0x00c4, 0x00c5, // 8x
	0x00c9, 0x00e6, 0x00c6, 0x00f4, 0x00f6, 0x00f2, 0x00fb, 0x00f9, 0x00ff, 0x00d6, 0x00dc, 0x00a2, 0x00a3, 0x00a5, 0x20a7, 0x0192, // 9x
	0x00e1, 0x00ed, 0x00f3, 0x00fa, 0x00f1, 0x00d1, 0x00aa, 0x00ba, 0x00bf, 0x2310, 0x00ac, 0x00bd, 0x00bc, 0x00a1, 0x00ab, 0x00bb, // ax
	0x2591, 0x2592, 0x2593, 0x2502, 0x2524, 0x2561, 0x2562, 0x2556, 0x2555, 0x2563, 0x2551, 0x2557, 0x255d, 0x255c, 0x255b, 0x2510, // bx
	0x2514, 0x2534, 0x252c, 0x251c, 0x2500, 0x253c, 0x255e, 0x255f, 0x255a, 0x2554, 0x2569, 0x2566, 0x2560, 0x2550, 0x256c, 0x2567, // cx
	0x2568, 0x2564, 0x2565, 0x2559, 0x2558, 0x2552, 0x2553, 0x256b, 0x256a, 0x2518, 0x250c, 0x2588, 0x2584, 0x258c, 0x2590, 0x2580, // dx
	0x03b1, 0x00df, 0x0393, 0x03c0, 0x03a3, 0x03c3, 0x00b5, 0x03c4, 0x03a6, 0x0398, 0x03a9, 0x03b4, 0x221e, 0x03c6, 0x03b5, 0x2229, // ex
	0x2261, 0x00b1, 0x2265, 0x2264, 0x2320, 0x2321, 0x00f7, 0x2248, 0x00b0, 0x2219, 0x00b7, 0x221a, 0x207f, 0x00b2, 0x25a0, 0x00a0  // fx
};

void utf8_to_cp437(const char *src, size_t srclen, char *dst, size_t dstlen)
{
	if (to_cp437_from_utf8 != (iconv_t)(-1))
	{
		while (*src && srclen && dstlen)
		{
			size_t res;

			res = iconv(to_cp437_from_utf8, (char **)&src, &srclen, &dst, &dstlen);

			if (res==(size_t)(-1))
			{
				if (errno==E2BIG) /* dstbuffer is full */
					break;
				if (errno==EILSEQ)
				{
					/* invalid input character, or we have a character that can't be translated, so skip it */
					int length = 0;
					utf8_decode (src, srclen, &length);
					src += length;
					srclen -= length;
					*dst = '?';
					dstlen--;
					continue;
				}
				break;
			}
		}
	}
	if (dstlen)
	{
		*dst = 0;
	}
	if (to_cp437_from_utf8 != (iconv_t)(-1))
	{
		iconv (to_cp437_from_utf8, 0, 0, 0, 0);
	}
}

void cp437_f_to_utf8_z(const char *src, size_t srclen, char *dst, size_t dstlen)
{
	char *eob = memchr (src, 0, srclen);
	if (eob)
	{
		srclen = eob - src;
	}
	if (!dstlen)
	{
		return;
	}
	dstlen--;
	if (from_cp437_to_utf8 != (iconv_t)(-1))
	{
		while (*src && srclen && dstlen)
		{
			size_t res;

			res = iconv(from_cp437_to_utf8, (char **)&src, &srclen, &dst, &dstlen);

			if (res==(size_t)(-1))
			{
				if (errno==E2BIG) /* dstbuffer is full */
					break;
				if (errno==EILSEQ)
				{
					src += 1;
					srclen -= 1;
					*dst = '?';
					dstlen--;
					continue;
				}
				break;
			}
		}
	}
	dstlen++;
	*dst = 0;
	if (from_cp437_to_utf8 != (iconv_t)(-1))
	{
		iconv (from_cp437_to_utf8, 0, 0, 0, 0);
	}
}


void  __attribute__((constructor)) cp437_charset_init(void)
{
	to_cp437_from_utf8 = iconv_open(OCP_FONT "//TRANSLIT", "UTF-8");
	if (to_cp437_from_utf8==(iconv_t)(-1))
	{
		fprintf(stderr, "iconv_open(\"%s\", \"UTF-8\") failed: %s - retrying \"%s\"\n", OCP_FONT "//TRANSLIT", strerror(errno), OCP_FONT);

		to_cp437_from_utf8 = iconv_open(OCP_FONT, "UTF-8");
		if (to_cp437_from_utf8==(iconv_t)(-1))
		{
			fprintf(stderr, "iconv_open(\"%s\", \"UTF-8\") failed: %s\n", OCP_FONT, strerror(errno));

			to_cp437_from_utf8 = iconv_open("CP850", "UTF-8");
			if (to_cp437_from_utf8==(iconv_t)(-1))
			{
				fprintf(stderr, "iconv_open(\"CP850\", \"UTF-8\") failed: %s\n", strerror(errno));

				to_cp437_from_utf8 = iconv_open("ASCII", "UTF-8");
				if (to_cp437_from_utf8==(iconv_t)(-1))
				{
					fprintf(stderr, "iconv_open(\"ASCII\", \"UTF-8\") failed: %s\n", strerror(errno));
				}
			}
		}
	}

	from_cp437_to_utf8 = iconv_open("UTF-8//TRANSLIT", OCP_FONT);
	if (from_cp437_to_utf8==(iconv_t)(-1))
	{
		fprintf(stderr, "iconv_open(\"UTF-8//TRANSLIT\", \"%s\") failed: %s - retrying \"UTF-8\"\n", OCP_FONT, strerror(errno));

		from_cp437_to_utf8 = iconv_open("UTF-8", OCP_FONT);
		if (from_cp437_to_utf8==(iconv_t)(-1))
		{
			fprintf(stderr, "iconv_open(\"UTF-8\", \"%s\") failed: %s\n", OCP_FONT, strerror(errno));

			from_cp437_to_utf8 = iconv_open("UTF-8", "CP850");
			if (from_cp437_to_utf8==(iconv_t)(-1))
			{
				fprintf(stderr, "iconv_open(\"UTF-8\", \"CP850\") failed: %s\n", strerror(errno));

				from_cp437_to_utf8 = iconv_open("UTF-8", "ASCII");
				if (from_cp437_to_utf8==(iconv_t)(-1))
				{
					fprintf(stderr, "iconv_open(\"UTF-8\", \"ASCII\") failed: %s\n", strerror(errno));
				}
			}
		}
	}
}

void  __attribute__((destructor)) cp437_charset_done(void)
{
	if (to_cp437_from_utf8 != (iconv_t)(-1))
	{
		iconv_close(to_cp437_from_utf8);
		to_cp437_from_utf8 = (iconv_t)(-1);
	}

	if (from_cp437_to_utf8 != (iconv_t)(-1))
	{
		iconv_close(from_cp437_to_utf8);
		from_cp437_to_utf8 = (iconv_t)(-1);
	}
}
